<?php
/*
Plugin Name: Error Log Cleaner
Description: Automatically deletes error_log files in /public_html and /public_html/wp-admin every Friday, with a manual cleanup option.
Version: 1.1
Author: goldlink.ir
License: GPL2
*/

// جلوگیری از دسترسی مستقیم به فایل
if (!defined('ABSPATH')) {
    exit;
}

class Error_Log_Cleaner {
    public function __construct() {
        // ثبت زمان‌بندی برای پاکسازی هفتگی
        add_action('wp', array($this, 'schedule_cleanup'));
        // ثبت اکشن برای اجرای پاکسازی
        add_action('error_log_cleanup_event', array($this, 'clean_error_logs'));
        // افزودن منوی تنظیمات به پیشخوان
        add_action('admin_menu', array($this, 'add_admin_menu'));
        // مدیریت درخواست پاکسازی دستی
        add_action('admin_init', array($this, 'handle_manual_cleanup'));
    }

    /**
     * زمان‌بندی پاکسازی فایل‌های error_log
     */
    public function schedule_cleanup() {
        if (!wp_next_scheduled('error_log_cleanup_event')) {
            // تنظیم زمان‌بندی برای هر جمعه ساعت 00:00
            wp_schedule_event(strtotime('next Friday 00:00'), 'weekly', 'error_log_cleanup_event');
        }
    }

    /**
     * پاک کردن فایل‌های error_log
     */
    public function clean_error_logs() {
        // مسیرهای مورد نظر
        $paths = array(
            ABSPATH . 'error_log', // مسیر /public_html/error_log
            ABSPATH . 'wp-admin/error_log' // مسیر /public_html/wp-admin/error_log
        );

        $deleted = false;
        foreach ($paths as $file_path) {
            // بررسی وجود فایل و پاک کردن آن
            if (file_exists($file_path)) {
                @unlink($file_path); // حذف فایل
                $deleted = true;
            }
        }

        return $deleted;
    }

    /**
     * افزودن منوی تنظیمات به پیشخوان
     */
    public function add_admin_menu() {
        add_options_page(
            'Error Log Cleaner',
            'Error Log Cleaner',
            'manage_options',
            'error-log-cleaner',
            array($this, 'render_admin_page')
        );
    }

    /**
     * رندر صفحه تنظیمات در پیشخوان
     */
    public function render_admin_page() {
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Error Log Cleaner', 'error-log-cleaner'); ?></h1>
            <p><?php esc_html_e('This plugin automatically deletes error_log files every Friday. You can also manually clean them using the button below.', 'error-log-cleaner'); ?></p>
            
            <?php
            // نمایش پیام موفقیت یا خطا
            if (isset($_GET['cleaned']) && $_GET['cleaned'] === '1') {
                echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Error log files have been successfully deleted.', 'error-log-cleaner') . '</p></div>';
            } elseif (isset($_GET['cleaned']) && $_GET['cleaned'] === '0') {
                echo '<div class="notice notice-info is-dismissible"><p>' . esc_html__('No error log files were found to delete.', 'error-log-cleaner') . '</p></div>';
            }
            ?>

            <form method="post" action="">
                <?php wp_nonce_field('error_log_cleanup_action', 'error_log_cleanup_nonce'); ?>
                <input type="hidden" name="manual_cleanup" value="1">
                <?php submit_button(__('Clean Error Logs Now', 'error-log-cleaner'), 'primary', 'submit', false); ?>
            </form>
        </div>
        <?php
    }

    /**
     * مدیریت درخواست پاکسازی دستی
     */
    public function handle_manual_cleanup() {
        if (isset($_POST['manual_cleanup']) && $_POST['manual_cleanup'] === '1') {
            // بررسی مجوز کاربر و nonce
            if (current_user_can('manage_options') && check_admin_referer('error_log_cleanup_action', 'error_log_cleanup_nonce')) {
                $deleted = $this->clean_error_logs();
                // هدایت به صفحه با پیام مناسب
                wp_redirect(add_query_arg('cleaned', $deleted ? '1' : '0', admin_url('options-general.php?page=error-log-cleaner')));
                exit;
            }
        }
    }

    /**
     * غیرفعال کردن زمان‌بندی هنگام غیرفعال کردن افزونه
     */
    public function deactivate() {
        wp_clear_scheduled_hook('error_log_cleanup_event');
    }
}

// ایجاد نمونه از کلاس
$error_log_cleaner = new Error_Log_Cleaner();

// ثبت تابع غیرفعال‌سازی
register_deactivation_hook(__FILE__, array($error_log_cleaner, 'deactivate'));